//==========================================
// Fridge
// Made for Persistence of vision 3.6
//==========================================  
// Copyright 2001-2004 Gilles Tran http://www.oyonale.com
// -----------------------------------------
// This work is licensed under the Creative Commons Attribution License. 
// To view a copy of this license, visit http://creativecommons.org/licenses/by/2.0/ 
// or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
// You are free:
// - to copy, distribute, display, and perform the work
// - to make derivative works
// - to make commercial use of the work
// Under the following conditions:
// - Attribution. You must give the original author credit.
// - For any reuse or distribution, you must make clear to others the license terms of this work.
// - Any of these conditions can be waived if you get permission from the copyright holder.
// Your fair use and other rights are in no way affected by the above. 
//==========================================  

#include "colors.inc"
#include "functions.inc"
#include "textures.inc"
#include "metals.inc"


// ----------------------------------------
global_settings{
    assumed_gamma 1
    radiosity{
//        count 200 error_bound 0.1
        recursion_limit 2
        brightness 2
        normal on
        adc_bailout 1/10
    }
}

camera {
  location  <0.0, 1, -4.0>
  direction 2*z
  right     x*image_width/image_height
  look_at   <0.0, 0.5,  -0.5>
  rotate y*40
}


//light_source {<30, 30, -30> color rgb <160,223,255>*0.2/255}

// ----------------------------------------
// Textures
// ----------------------------------------
// rubber                
#declare T_Joint=texture{
    pigment{rgb<0.99,1,1>*0.7} 
    normal{bumps 0.3 scale 10} 
    finish{ambient 0 diffuse 0.5 specular 0.2 roughness 0.1}
}

// fridge paint
#declare T_Fridge=texture{
    pigment{rgb<0.99,1,1>} 
    finish{ambient 0 diffuse 0.8 specular 1 roughness 0.001 reflection 0.4}
    normal{bozo 0.1 scale 50}
}

// plastic
#declare T_FridgePlastic=texture{
    pigment{rgb<0.99,1,1>*1.4} 
    normal{bozo 0.5 scale 50}
    finish{ambient 0 diffuse 0.5 specular 0.1 roughness 0.1 reflection 0.05}
}

// leg
#declare T_Leg=texture{
    pigment{rgb<0.99,1,1>*0.3} 
    finish{ambient 0 diffuse 0.8 specular 0.3 roughness 0.1 metallic brilliance 7 reflection 0.1}
}

// various
#declare T_Ground=texture{T_Fridge}
#declare T_Wall=texture{pigment{rgb<0.95,1,1>} finish{ambient 0 diffuse 0.8 specular 1 roughness 0.1 }}

#declare T_Fume=material{
    texture{pigment{rgbf<0.9,0.85,0.8,0.9>} finish{ambient 0 diffuse 0.6 reflection 0.1}}
    interior{ior 1.2}
}              
#declare T_Fume=material{texture{T_Fridge}}


#declare T_Star=texture{
        pigment{White*1.5}
        finish{ambient 0 diffuse 1 brilliance 4 specular 1 roughness 0.05 reflection {0.2 metallic}}
}

// handle
#declare T_Handle=texture{T_Silver_5E}

// ----------------------------------------
// Fridge
// ----------------------------------------
// height field used to put the inner shelves on
#declare Side0=height_field{jpeg "hfrainure.jpg"  smooth rotate x*-90 rotate y*-90 scale <0.3,1,1> translate x}
#declare i=0;              
                             
#declare n=5;
#declare rd=seed(1);
#declare Side=union{
    #while (i<n)
        object{Side0 scale <1,1/n,1> translate y*i/n} // pigment{color rgb<rand(rd),rand(rd),rand(rd)>}}
        #declare i=i+1;
    #end
} 

// little star on handle
#declare Star=union{
    #declare i=0;
    #while (i<360)
        triangle{-z*1.3,x,<cos(pi/5),cos(pi/5)/tan(pi/10),0> rotate -z*i}
        triangle{-z*1.3,x,<cos(pi/5),-cos(pi/5)/tan(pi/10),0> rotate -z*i}
        #declare i=i+72;
    #end
    rotate z*18
}                          


// handle
#declare Handle=union{
    union{
        object{Star}
        object{Star translate x*6}
        object{Star translate x*12}
        scale 0.04
        translate <-0.32,0.61,-0.95>
        texture{T_Star}
    }                
    difference{
        superellipsoid{<0.1,0.6>}
        superellipsoid{<0.2,0.2> scale <1,0.1,0.2> translate <0.5,0.3,-1.12>}
        superellipsoid{<0.2,0.2> scale <1,0.1,0.2> translate <0.5,0,-1.12>}
        superellipsoid{<0.2,0.2> scale <1,0.1,0.2> translate <0.5,-0.3,-1.12>}
        cylinder{-y*1.4,y*1.4,1 translate <0.3,0,0.2>}
        cylinder{-z*1.4,z*1.4,2 scale <0.7,1,1> translate <-1,1,0> inverse}
        scale <1,2,1>
    }        
    texture{T_Handle}
}

// round box macro
#macro RoundedBox(bdown,bup,br)
union{
    sphere{0,br translate bdown+br}
    sphere{0,br translate <bdown.x+br,bup.y-br,bdown.z+br>}
    sphere{0,br translate <bup.x-br,bup.y-br,bdown.z+br>}
    sphere{0,br translate <bup.x-br,bdown.y+br,bdown.z+br>}
    sphere{0,br translate bup-br}
    sphere{0,br translate <bdown.x+br,bup.y-br,bup.z-br>}
    sphere{0,br translate <bdown.x+br,bdown.y+br,bup.z-br>}
    sphere{0,br translate <bup.x-br,bdown.y+br,bup.z-br>}
    box{<bdown.x+br,bdown.y+br,bdown.z>,<bup.x-br,bup.y-br,bdown.z+br>}
    box{<bdown.x,bdown.y+br,bdown.z+br>,<bdown.x+br,bup.y-br,bup.z-br>}
    box{<bup.x-br,bdown.y+br,bdown.z+br>,<bup.x,bup.y-br,bup.z-br>}
    box{<bdown.x+br,bup.y-br,bdown.z+br>,<bup.x-br,bup.y,bup.z-br>}
    box{<bdown.x+br,bdown.y,bdown.z+br>,<bup.x-br,bdown.y+br,bup.z-br>}
    box{<bdown.x+br,bdown.y+br,bup.z-br>,<bup.x-br,bup.y-br,bup.z>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bdown.y+br,bdown.z+br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bup.y-br,bdown.z+br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bdown.y+br,bup.z-br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bup.y-br,bup.z-br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bdown.x+br,0,bdown.z+br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bup.x-br,0,bdown.z+br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bdown.x+br,0,bup.z-br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bup.x-br,0,bup.z-br>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bdown.x+br,bdown.y+br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bdown.x+br,bup.y-br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bup.x-br,bdown.y+br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bup.x-br,bup.y-br,0>}
}        
#end

// bottomless rounded box macro
#macro RoundedBoxNoBottom(bdown,bup,br)
union{
    sphere{0,br translate bdown+br}
    sphere{0,br translate <bdown.x+br,bup.y-br,bdown.z+br>}
    sphere{0,br translate <bup.x-br,bup.y-br,bdown.z+br>}
    sphere{0,br translate <bup.x-br,bdown.y+br,bdown.z+br>}
    sphere{0,br translate bup-br}
    sphere{0,br translate <bdown.x+br,bup.y-br,bup.z-br>}
    sphere{0,br translate <bdown.x+br,bdown.y+br,bup.z-br>}
    sphere{0,br translate <bup.x-br,bdown.y+br,bup.z-br>}
    box{<bdown.x+br,bdown.y+br,bdown.z>,<bup.x-br,bup.y-br,bdown.z+br>}
    box{<bdown.x,bdown.y+br,bdown.z+br>,<bdown.x+br,bup.y-br,bup.z-br>}
    box{<bup.x-br,bdown.y+br,bdown.z+br>,<bup.x,bup.y-br,bup.z-br>}
    box{<bdown.x+br,bup.y-br,bdown.z+br>,<bup.x-br,bup.y,bup.z-br>}
    box{<bdown.x+br,bdown.y,bdown.z+br>,<bup.x-br,bdown.y+br,bup.z-br>}
//        box{<bdown.x+br,bdown.y+br,bup.z-br>,<bup.x-br,bup.y-br,bup.z>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bdown.y+br,bdown.z+br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bup.y-br,bdown.z+br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bdown.y+br,bup.z-br>}
    cylinder{x*(bdown.x+br),x*(bup.x-br),br translate <0,bup.y-br,bup.z-br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bdown.x+br,0,bdown.z+br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bup.x-br,0,bdown.z+br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bdown.x+br,0,bup.z-br>}
    cylinder{y*(bdown.y+br),y*(bup.y-br),br translate <bup.x-br,0,bup.z-br>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bdown.x+br,bdown.y+br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bdown.x+br,bup.y-br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bup.x-br,bdown.y+br,0>}
    cylinder{z*(bdown.z+br),z*(bup.z-br),br translate <bup.x-br,bup.y-br,0>}
}        
#end


// outer fridge box
#declare yLeg=10;
#declare FridgeBox=union{
    #declare xFe=60;
    #declare yFe=100;
    #declare zFe=50;
    #declare eF=4;
    #declare rF=2;
    #declare eFa=eF*0.5;
    #declare rFa=(eF-eFa)*0.5;
    #declare rFa1=rFa*0.98;
    #declare xFi=xFe-2*eF-2*rF;
    #declare yFi=yFe-2*eF-2*rF;
    #declare zFi=zFe-eF-rF;
    
    // grid shelf
    #declare Grid=union{
        #declare xG=xFe-2*eF;
        #declare zG=zFe;
        #declare nG=15;
        #declare rG=0.3;
        #declare i=1;
        #while (i<nG)
            union{
                cylinder{0,z*zG,rG}
                sphere{0,rG scale <1,1,0.3>}
                translate x*i*xG/nG
            }
            #declare i=i+1;
        #end
        cylinder{0,x*xG,rG*1.1 translate <0,0,rG*2>}
        cylinder{0,x*xG,rG*1.1 translate <0,0,zG*0.5>}
        cylinder{0,x*xG,rG*1.1 translate <0,0,zG-rG*2>}
        translate y*rG*3
        texture{T_FridgePlastic}
    }
    
    // corner
    #declare Corner=union{
        difference{
            box{0,1}
            cylinder{-z*1.1,z*1.1,0.8 translate y}
        }                  
        difference{
            sphere{0,1}
            cylinder{-y*2,y*2,0.2 translate -z rotate z*68}
            scale <0.25,0.25,0.02> translate <0.65,0.25,0>
        }
        translate -x
        scale <rF+eF,rF+eF,zFi>
    }
    
    #declare nF=6;
    union{ // cts
        object{Corner translate <xFe,-eF-rF,-eF*0.3>}
        object{Corner scale <-1,1,1> translate <0,-eF-rF,-eF*0.3>}
        object{Corner scale <1,-1,1> translate <xFe,yFi+eF+rF,-eF*0.3>}
        object{Corner scale <-1,-1,1> translate <0,yFi+eF+rF,-eF*0.3>}
        

        box{0,<eF*0.3,yFi,zFi>}
        box{0,<eF*0.3,yFi,zFi> scale <-1,1,1> translate x*xFe}
        object{Side scale <eF,yFi,zFi>}
        object{Side scale <eF,yFi,zFi> scale <-1,1,1> translate x*xFe}
        object{Grid translate <eF,yFi*1.5/5,rF>}
        object{Grid translate <eF,yFi*3.5/5,rF>}


        #declare sc=28;

        
        box{0,<xFi,yFi,eF> translate <eF+rF,0,zFi+rF>}
        union{
            difference{cylinder{0,y*yFi,eF+rF} cylinder{-y*0.1,y*yFi*1.1,rF} plane{x,0 inverse} plane{z,0} translate x*(rF+eF)}
            difference{cylinder{0,y*yFi,eF+rF} cylinder{-y*0.1,y*yFi*1.1,rF}  plane{x,0 inverse} plane{z,0} scale <-1,1,1> translate x*(eF+xFi+rF)}
            translate z*zFi
        }     
    }        
    
    // roof      
    union{ //Toit
        box{0,<xFi,eF,zFi> translate <eF+rF,rF,0>}
        difference{cylinder{0,z*zFi,rF+eF}cylinder{-z*0.1,z*zFi*1.1,rF} plane{y,0}plane{x,0 inverse} translate x*(rF+eF)}
        difference{cylinder{0,z*zFi,rF+eF}cylinder{-z*0.1,z*zFi*1.1,rF} plane{y,0}plane{x,0 inverse} scale <-1,1,1> translate x*(eF+xFi+rF)}
        difference{cylinder{0,x*xFi,rF+eF}cylinder{-x*0.1,x*xFi*1.1,rF} plane{y,0}plane{z,0} translate <eF+rF,0,zFi>}
        difference{sphere{0,rF+eF}sphere{0,rF} plane{y,0}plane{z,0}plane{x,0 inverse} translate <eF+rF,0,zFi>}
        difference{sphere{0,rF+eF}sphere{0,rF} plane{y,0}plane{z,0}plane{x,0 inverse} scale <-1,1,1> translate <eF+xFi+rF,0,zFi>}
        translate <0,yFi,0>
    }
    
    // ground
    union{ //Sol
        box{0,<xFi,eF,zFi> translate <eF+rF,rF,0>}
        difference{cylinder{0,z*zFi,rF+eF}cylinder{-z*0.1,z*zFi*1.1,rF} plane{y,0}plane{x,0 inverse} translate x*(rF+eF)}
        difference{cylinder{0,z*zFi,rF+eF}cylinder{-z*0.1,z*zFi*1.1,rF} plane{y,0}plane{x,0 inverse} scale <-1,1,1> translate x*(eF+xFi+rF)}
        difference{cylinder{0,x*xFi,rF+eF}cylinder{-x*0.1,x*xFi*1.1,rF} plane{y,0}plane{z,0} translate <eF+rF,0,zFi>}
        difference{sphere{0,rF+eF}sphere{0,rF} plane{y,0}plane{z,0}plane{x,0 inverse} translate <eF+rF,0,zFi>}
        difference{sphere{0,rF+eF}sphere{0,rF} plane{y,0}plane{z,0}plane{x,0 inverse} scale <-1,1,1> translate <eF+xFi+rF,0,zFi>}
        scale <1,-1,1>
    }

    // left
    union{ // bord verticaux gauche
        box{0,<eFa,yFi,rFa1> scale <1,1,-1>}
        cylinder{0,yFi*y,rFa}
        cylinder{0,yFi*y,rFa translate x*eFa}
        translate x*rFa
    }      
    
    // right
    union{ // bord verticaux droite
        box{0,<eFa,yFi,rFa1> scale <1,1,-1>}
        cylinder{0,yFi*y,rFa}
        cylinder{0,yFi*y,rFa translate x*eFa}
        translate x*(eF+2*rF+xFi+rFa)
    }       
    
    // top
    union{ // bord sommet
        box{0,<xFi,eFa,rFa1> scale <1,1,-1>}
        cylinder{0,x*xFi,rFa}
        cylinder{0,x*xFi,rFa translate y*eFa}
        union{
            difference{torus{rF+eFa+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{torus{rF+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{cylinder{-z*rFa1,0,rF+rFa+eFa} cylinder{-z*rFa*1.1,z*0.1,rF+rFa} plane{y,0}plane{x,0 inverse}}
            translate -y*(rF+rFa)
        }
        union{
            difference{torus{rF+eFa+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{torus{rF+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{cylinder{-z*rFa1,0,rF+rFa+eFa} cylinder{-z*rFa*1.1,z*0.1,rF+rFa} plane{y,0}plane{x,0 inverse}}
            scale <-1,1,1>
            translate <xFi,-rF-rFa,0>
        }
        translate <eF+rF,yFi+rF+rFa,0>
    }     
    
    // bottom
    union{ // bord base
        box{0,<xFi,eFa,rFa1> scale <1,1,-1>}
        cylinder{0,x*xFi,rFa}
        cylinder{0,x*xFi,rFa translate y*eFa}
        union{
            difference{torus{rF+eFa+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{torus{rF+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{cylinder{-z*rFa1,0,rF+rFa+eFa} cylinder{-z*rFa*1.1,z*0.1,rF+rFa} plane{y,0}plane{x,0 inverse}}
            translate -y*(rF+rFa)
        }
        union{
            difference{torus{rF+eFa+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{torus{rF+rFa,rFa rotate x*90}plane{y,0}plane{x,0 inverse}}
            difference{cylinder{-z*rFa1,0,rF+rFa+eFa} cylinder{-z*rFa*1.1,z*0.1,rF+rFa} plane{y,0}plane{x,0 inverse}}
            scale <-1,1,1>
            translate <xFi,-rF-rFa,0>
        }
        scale <1,-1,1>
        translate <eF+rF,-rFa-rF,0>
            
    }              
    translate y*(eF+rF)
    
}                                                              

// magnet
#declare Magnet=union{
    cylinder{-z*0.9,0,1}
    torus{0.9,0.1 rotate x*90 translate -z*0.9}
    cylinder{-z,0,0.9
        texture{
            // little cow = logotype of the French dairy information centre
            pigment{image_map{png "cidil"} translate <-0.5,-0.5,0> scale 2}
            finish{ambient 0 diffuse 0.9 specular 1 roughness 0.1 reflection 0.3}
        }                
    }
    scale 2
}

// leg
#declare Leg=union{
    difference{
        superellipsoid{<0.1,0.1>}
        superellipsoid{<0.1,0.1> scale <0.9,0.12,1> translate <0,0.5,-0.2>}
        superellipsoid{<0.1,0.1> scale <0.9,0.12,1> translate <0,0,-0.2>}
        superellipsoid{<0.1,0.1> scale <0.9,0.12,1> translate <0,-0.5,-0.2>}
    }           
    #declare i=-1;
    #declare n=5;
    #declare rd=seed(0);
    #while (i<1-1/n) 
    #declare i=i+1/n;
        superellipsoid{<0.1,0.1> rotate z*(0.5-rand(rd))*10 scale <0.01,0.7,1> translate <i,0,-0.03>}
    #end
    translate 1 scale 0.5*<xFe*1.1,yLeg,zFe> 
    translate <-xFe*0.05,0,eF>
}                                                              


// fridge door
// took me several days to get it right
#declare yFreezer=40;                                                                   
#declare eDoor=eF*2;
#declare yDoor=100;                                        
#declare yPi=yFe-eF*0.3;
#declare xPi=xFe+3*eF;

// rubber
#declare JointElement=prism{
	cubic_spline
	0,	1.000000,	20,
	<0.000000,0.922275>,	<0.000000,0.746269>,	<0.000000,0.218248>,	<0.035201,0.007040>,	<0.246409,0.042242>,	<0.352013,0.007040>,	<0.457618,0.077443>,
	<0.422416,0.253450>,	<0.457618,0.394255>,	<0.316812,0.464658>,	<0.281611,0.570262>,	<0.281611,0.675866>,	<0.387215,0.711067>,
	<0.387215,0.957477>,	<0.211208,0.957477>,	<0.070403,0.992678>,	<0.000000,0.992678>,	<0.000000,0.922275>,	<0.000000,0.746269>,	<0.000000,0.218248>
}

#declare Joint=union{
    #declare sJoint=5;
    #declare xJoint=xPi-sJoint;
    #declare yJoint=yPi-sJoint;
    #declare eJoint=5;
    #declare zJoint=5;
    #declare JointBas=difference{
        object{JointElement rotate y*-90 rotate z*-90scale <xJoint,eJoint,zJoint>}
        plane{x,0 rotate -z*45}
        plane{x,0 rotate z*45 inverse translate x*xJoint}
    }
    #declare JointSide=difference{
        object{JointElement rotate y*-90 scale <-eJoint,yJoint,zJoint>}
        plane{y,0 rotate z*45}
        plane{y,0 rotate -z*45 translate y*yJoint inverse}
    }
    object{JointBas}        
    object{JointBas scale <1,-1,1> translate y*yJoint}        
    object{JointSide}        
    object{JointSide scale <-1,1,1> translate x*xJoint}        
    texture{T_Joint}
}

// inner frame of the door
#declare sCi=eJoint+sJoint/2;
#declare xCi=xJoint-eJoint*2;
#declare zCi=12;
#declare zCi1=5;
#declare rCi=5;
#declare rCi1=3;
#declare rCi2=1;
#declare eCi=2;          
#declare reCi=eCi*0.5;                                 
#declare reCib=reCi*0.6;
#declare yCi=yJoint-2*(eJoint+rCi+rCi1+rCi2);      
#declare xCib=xCi-2*eCi;
#declare yCi1=yCi*0.15;
#declare yCi3=yCi*0.3;
#declare yCi2=yCi-yCi1-yCi3;
#declare FrameSide=union{          
    union{
        difference{
            cylinder{0,z*zCi,rCi}
            cylinder{-z,z*zCi*1.1,rCi-eCi}
            plane{x,0 inverse}
            plane{y,0 inverse}
        }
        difference{
            torus{rCi-reCi,reCi rotate x*90 translate z*zCi}
            plane{x,0 inverse}
            plane{y,0 inverse}
        }                
        translate <rCi,rCi,0>
    }
    #declare ytemp=rCi;
    union{                       
        box{<0,0,0>,<eCi,yCi1,zCi>}
        cylinder{0,y*yCi1,reCi translate <reCi,0,zCi>}
        translate y*ytemp
    }
    #declare ytemp=ytemp+yCi1;
    union{
        difference{
            box{<0,0,0>,<eCi,rCi1*2,zCi-rCi1>}
            cylinder{-x,x*eCi*2,rCi1 translate <0,rCi1*2,zCi-rCi1>}
        }                        
        cylinder{0,x*eCi,rCi1 translate <0,0,zCi-rCi1>}
        torus{rCi1,reCi rotate z*90 translate <reCi,0,zCi-rCi1>}
        translate y*ytemp
    }
    #declare ytemp=ytemp+rCi1;
    difference{
        torus{rCi1,reCi rotate z*90}
        plane{y,0 inverse}
        plane{z,0 inverse}       
        translate <reCi,rCi1,zCi-rCi1>
        translate y*ytemp        
            
    }
    #declare ytemp=ytemp+rCi1;
    union{                       
        box{<0,0,0>,<eCi,yCi2,zCi-rCi1*2>}
        cylinder{0,y*yCi2,reCi translate <reCi,0,zCi-rCi1*2>}
        translate y*ytemp
    }
    #declare ytemp=ytemp+yCi2;
    
    union{
        difference{
            box{<0,0,0>,<eCi,rCi2*2,zCi-rCi1*2-rCi2>}
            cylinder{-x,x*eCi*2,rCi2 translate <0,rCi2*2,zCi-rCi1*2-rCi2>}
        }                        
        cylinder{0,x*eCi,rCi2 translate <0,0,zCi-rCi1*2-rCi2>}
        torus{rCi2,reCi rotate z*90 translate <reCi,0,zCi-rCi1*2-rCi2>}
        translate y*ytemp
    }
    #declare ytemp=ytemp+rCi2;     
    
    difference{
        torus{rCi2,reCi rotate z*90}
        plane{y,0 inverse}
        plane{z,0 inverse}       
        translate <reCi,rCi2,zCi-rCi1*2-rCi2>
        translate y*ytemp        
        
    }
    #declare ytemp=ytemp+rCi2;
    union{                       
        box{<0,0,0>,<eCi,yCi3,zCi-rCi1*2-rCi2*2>}
        cylinder{0,y*yCi3,reCi translate <reCi,0,zCi-rCi1*2-rCi2*2>}
        translate y*ytemp
    }
    #declare yCi4=ytemp+yCi3;


    union{
        difference{
            cylinder{0,z*(zCi-rCi1*2-rCi2*2),rCi}
            cylinder{-z,z*(zCi-rCi1*2-rCi2*2)*1.1,rCi-eCi}
            plane{x,0 inverse}
            plane{y,0}
        }
        difference{
            torus{rCi-reCi,reCi rotate x*90 translate z*(zCi-rCi1*2-rCi2*2)}
            plane{x,0 inverse}
            plane{y,0}
        }                
        translate <rCi,yCi4,0>
    }
    
}
#declare FrameInterior=union{
    #declare xCi1=xCi-2*rCi;
    object{FrameSide}
    object{FrameSide scale <-1,1,1> translate <xCi,0,0>}
    union{
        box{0,<xCi1,eCi,zCi>}
        cylinder{0,x*xCi1,reCi translate<0,reCi,zCi>}
        translate x*rCi
    }
    union{
        box{0,<xCi1,eCi,zCi-2*(rCi1+rCi2)>}
        cylinder{0,x*xCi1,reCi translate<0,reCi,zCi-2*(rCi1+rCi2)>}
        scale <1,-1,1>
        translate <rCi,yCi4+rCi,0>
    }
    difference{
        cylinder{0,x*xCi1,rCi}
        cylinder{-x,x*xCi1*1.1,rCi-eCi}
        plane{y,0 inverse}
        plane{z,0 inverse}
        translate <rCi,rCi,0>
    }
    difference{
        cylinder{0,x*xCi1,rCi}
        cylinder{-x,x*xCi1*1.1,rCi-eCi}
        plane{y,0 inverse}
        plane{z,0 inverse}
        translate <rCi,rCi,0>
        scale <1,-1,1>
        translate y*(yCi4+rCi)
    }            
    difference{
        cylinder{0,y*(yCi+2*(rCi1+rCi2)),rCi}
        cylinder{-y,1.1*y*(yCi+2*(rCi1+rCi2)),rCi-eCi}
        plane{x,0 inverse}
        plane{z,0 inverse}
        translate <rCi,rCi,0>
    }
    difference{
        cylinder{0,y*(yCi+2*(rCi1+rCi2)),rCi}
        cylinder{-y,1.1*y*(yCi+2*(rCi1+rCi2)),rCi-eCi}
        plane{x,0 inverse}
        plane{z,0 inverse}
        translate <rCi,rCi,0>
        scale <-1,1,1>
        translate x*xCi
    }
    #declare BarSupport=union{
        sphere{0,reCib scale <0.5,1,1>}
        sphere{0,reCib scale <0.5,1,1> translate x*eCi}
        sphere{0,reCib scale <0.5,1,1> translate x*(xCi-eCi)}
        sphere{0,reCib scale <0.5,1,1> translate x*xCi}
        cylinder{0,x*xCi,reCib*0.5}
        texture{T_FridgePlastic}
    }
    union{
        object{BarSupport}
        object{BarSupport translate y*(yCi1*0.33)}
        object{BarSupport translate y*(yCi1*0.67)}
        object{BarSupport translate y*(yCi1)}
//                translate <rCi+eCi,rCi,zCi+rCi+eCi+reCi>
        translate <0,rCi,zCi>
    }
    #declare FrameCorner=difference{
        sphere{0,rCi}
        sphere{0,rCi-eCi}
        plane{y,0 inverse}
        plane{z,0 inverse}
        plane{x,0 inverse}
    }
    object{FrameCorner translate <rCi,rCi,0>}
    object{FrameCorner scale <-1,1,1> translate <rCi+xCi1,rCi,0>}
    object{FrameCorner translate <rCi,rCi,0> scale <1,-1,1> translate y*(yCi4+rCi)}
    object{FrameCorner scale <-1,1,1> translate <rCi+xCi1,rCi,0> scale <1,-1,1> translate y*(yCi4+rCi)}
    box{0,<xCi1,yCi4,rCi-eCi> scale <1,1,-1> translate <rCi,rCi,-rCi+eCi>}
}
#declare xCs=xCi-2*eCi;
#declare zCs=7;
#declare eCs=1;          
#declare reCs=eCs*0.5;                                 
#declare reCsb=reCs*0.6;                                 
#declare rCs=eCs*2;
#declare rC1s=rCs;
#declare xC1s=xCs-2*rCs;
#declare aCs=30;
#declare yC1s=5;
#declare xCsb=xC1s-2*eCs;
#declare SupportSide=union{
    box{0,<eCs,yC1s,zCs>}

    cylinder{0,z*zCs,reCs translate <reCs,0,0>}
    difference{
        cylinder{0,z*zCs,rCs}
        cylinder{-z*0.1,z*zCs*1.1,rCs-eCs}
        plane{y,0 inverse}
        plane{x,0 inverse}     
        translate x*rCs
        
    }            
    difference{
        sphere{0,rCs}
        sphere{0,rCs-eCs}
        plane{y,0 inverse}
        plane{x,0 inverse}     
        plane{z,0} 
        translate <rCs,0,zCs>    
    }
    union{
        difference{
            cylinder{0,y*yC1s,rCs}
            cylinder{-y*0.1,y*yC1s*1.1,rCs-eCs}
            plane{z,0}
            plane{x,0 inverse}
            
        }
        difference{
            torus{rCs-reCs,reCs}
            plane{z,0}
            plane{x,0 inverse}
            translate y*yC1s
        }
        difference{
            torus{rCs-reCs,reCs}
            plane{z,0}
            plane{x,0 inverse}
        }                     
        cylinder{0,yC1s*y,reCs translate <0,0,rCs-reCs>}
        sphere{0,reCs translate <0,yC1s,rCs-reCs>}
        difference{
            torus{rCs-reCs,reCs rotate z*90}
            plane{y,0 inverse}
            plane{z,0}
        }
        translate <rCs,0,zCs>
    }
    #declare zC2s=zCs-rC1s*sin(radians(aCs));
    #declare yC2s=rC1s*(1-cos(radians(aCs)));
    #declare yC3s=zC2s*tan(radians(aCs));
            
    union{
        difference{
            torus{rC1s,reCs rotate z*90}
            plane{z,0 inverse}
            plane{z,0 rotate x*aCs}
            translate <0,rC1s,0>
        }
        cylinder{0,-z*zC2s/cos(radians(aCs)),reCs translate -y*rC1s rotate x*aCs translate y*rC1s}
        translate <reCs,yC1s,zCs>
    }
    box{0,<eCs,yC2s,zC2s> translate y*yC1s}
    difference{
        box{<0,0,-zCs+zC2s>,<eCs,yC2s,0>}
        cylinder{-x,x*eCs*1.1,rC1s translate <0,rC1s,0>}
        translate <0,yC1s,zCs>
    }
    difference{
        box{0,<eCs,yC3s,zC2s>}
        plane{y,0 inverse rotate x*aCs translate z*zC2s}
        translate <0,yC1s+yC2s,0>
    }                    
    #declare yC4s=yC1s+yC2s+yC3s;
    difference{
        sphere{0,reCs}
        plane{z,0}
        translate <reCs,yC4s,0>
    }


    union{
        difference{
            cylinder{0,y*yC4s,rCs}
            cylinder{-y*0.1,y*yC4s*1.1,rCs-eCs}
            plane{z,0 inverse}
            plane{x,0 inverse}
            
        }
        difference{
            torus{rCs-reCs,reCs}
            plane{z,0 inverse}
            plane{x,0 inverse}
            translate y*yC4s
        }                        
        difference{
            sphere{0,rCs}
            sphere{0,rCs-eCs}
            plane{y,0 inverse}
            plane{x,0 inverse}     
            plane{z,0 inverse} 
        }
        translate <rCs,0,0>
    }
    translate <reCs,rCs,rCs>
}                         
#declare BarSupport=union{
    sphere{0,reCsb}
    sphere{0,reCsb translate x*xCsb}
    cylinder{0,x*xCsb,reCsb*0.5}
}
#declare Support=union{
        object{SupportSide}
        object{SupportSide scale <-1,1,1> translate x*xCs}
        union{
            box{0,<xC1s,eCs,zCs>}
            cylinder{0,x*xC1s,reCs translate <0,reCs,zCs>}
            translate <rCs,0,rCs>
        }
        difference{
            cylinder{0,xC1s*x,rCs}
            cylinder{-x,xC1s*x*1.1,eCs}
            plane{y,0 inverse}
            plane{z,0 inverse}
            translate <rCs,rCs,rCs>
        }        
        box{0,<xC1s,yC4s,eCs> translate <rCs,rCs,0>}
        cylinder{0,x*xC1s,reCs translate <rCs,yC4s+rCs,reCs>}
        union{
            object{BarSupport}
            object{BarSupport translate y*(yC1s*0.33)}
            object{BarSupport translate y*(yC1s*0.67)}
            object{BarSupport translate y*(yC1s)}
            translate <rCs+eCs,rCs,zCs+rCs+eCs+reCs>
        }
        texture{T_FridgePlastic}
}

// now we fill the door shelves with objects
#declare FrameInteriorFull=union{
    object{FrameInterior}
    
    #include "bouteilles.inc"
    union{
        object{Bouteille2 scale <1,-1,1> translate y*0.76 scale 25/0.76 translate <8,0,8>}
        object{Bouteille4 scale <1,-1,1> translate y*0.76 rotate y*90 scale 24/0.76 translate <20,0,9>}
        object{Bouteille1 rotate y*170 scale <1,-1,1> translate y*0.76 scale 27/0.76 translate <35,0,6.5>}
        object{Bouteille3 scale <1,-1,1> translate y*0.76 scale 28/0.76 translate <50,0,6>}
//                photons {target 1.0 refraction on reflection on} // reflection on}
        translate y*eCi
    }

    #include "bocaux.inc"
    union{
        object{Support}
        
        object{BoutLait scale 0.8 rotate -y*5 translate <12,eCs,zCs*0.5>}
        object{BoutLait scale 0.8 rotate y*95 translate <22,eCs,zCs*0.45>}
        object{PotHexa scale 0.85 rotate y*95 translate <35,eCs,zCs*0.5>}
        
        translate <eCi,yCi*0.6,-rCi*0.7>
    }
    
    union{
        object{Support}
        object{Oeufs scale 0.45 translate <xCs-eCs*2,eCs,zCs*0.5>}

        translate <eCi,yCi*1,-rCi*0.7>
    }
    texture{T_Fridge}
}

// complete inner door
#declare DoorInterior=union{
    object{Joint translate <sJoint/2,sJoint/2,-1>}
    object{FrameInteriorFull translate <sCi,sCi,0>}
}

// complete door
// the yDoor value controls the aperture
#declare Door=union{
    object{RoundedBoxNoBottom(<-eF*1.5,0,-eDoor>,<xFe+eF*1.5,yFe-eF*0.3,0>,2)}
    union{
        cylinder{-y*eF,y*(yFe+eF*2),eF*0.4}
        torus{eF*0.4,eF*0.1 translate -y*eF texture{T_Joint}}
        translate <xFe+eF*0.75,0,-eDoor*0.5>
        texture{T_Handle}
    }
    object{Handle scale <1,1,0.7> scale yFreezer*0.4/2 translate <0,yFe*0.7,-eF*2>}
    
    object{DoorInterior translate <-eF*1.5,0,0>}
    
    translate <-xFe-eF*0.75,0,eDoor*0.5>
    rotate -y*yDoor
    translate <xFe+eF*0.75,0,-eDoor*0.5>
}           

// freezer door                                   
#declare DoorFreezer=union{
        object{RoundedBox(<-eF*1.5,0,-eDoor>,<xFe+eF*1.5,yFreezer,0>,2)}
        object{Handle scale <1,1,0.7> scale yFreezer*0.4/2 translate <0,yFreezer*0.5,-eF*2>}
        text{ttf "cyrvetic.ttf", "CoolCows",1,0.2*x scale 4 translate <xFe*0.5,yFreezer*0.1,-eDoor*1.1> texture{T_Star}}
        object{Magnet rotate z*10 translate <xFe*0.2,yFreezer*0.2,-eDoor>}
        object{Magnet rotate -z*30 translate <xFe*0.45,yFreezer*0.4,-eDoor>}
}        

// complete fridge                                   
#declare Fridge=union{

    object{RoundedBox(<-eF*1.5,0,0>,<0,yFe+yLeg+yFreezer,zFe>,2)}
    
    object{RoundedBox(<0,0,0>,<eF*1.5,yFe+yLeg+yFreezer,zFe>,2) translate x*xFe}
    box{0,<xFe,yFreezer,zFe> translate <0,yLeg+yFe-eF*0.2,0>}       
    object{DoorFreezer translate <0,yLeg+yFe,0>}
    object{FridgeBox translate <0,yLeg,eF*0.5> texture{T_FridgePlastic}}

    object{Door translate y*yLeg}

    // interior light
    light_source {<xFi*0.2,yLeg+yFi,zFi*0.8> rgb <1,0.8,0.4>
        fade_power 2 fade_distance 10
        area_light x*6,z*6,7,7 jitter adaptive 2
    }
    object{Leg texture{T_Leg}}
    texture{T_Fridge}
    translate -z*50
    scale 0.011
}

// ----------------------------------------
// scene
// ----------------------------------------
object{Fridge translate -x*0.95}
plane{y,0 
    texture{
        pigment{
            boxed
            color_map{
                [1/20 Black]
                [1/20  rgb <0.8,0.85,0.95>]
            }
            translate x+z
            warp{repeat x*2}
            warp{repeat z*2}
            scale 0.1
        }
        finish{ambient 0 diffuse 1}
    }
}         
plane{z,0 
    texture{
        pigment{
            boxed
            color_map{
                [1/20 Black]
                [1/20  rgb <0.8,0.85,0.95>]
            }
            translate x+y
            warp{repeat x*2}
            warp{repeat y*2}
            scale 0.1
        }
        finish{ambient 0 diffuse 1}
    }
    translate z*0.1
}         

plane{z,-5  hollow
    texture{
        pigment{
            rgb <160,223,255>*0.1/255
        }
        finish{ambient 1 diffuse 0}
    }
}         
